using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carrie Van Stedum</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>06/27/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// PatientUnitSelection data access layer class.
	///</summary>

	#endregion
	
	public class PatientUnitSelection
	{

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1553"> 
		///		<ExpectedInput>BloodTypeCode, rhFactorCode, PatientGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable of ProductShipper data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1554"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Selects "Whole Blood"  blood units that are compatible for a given patients abo rh
		/// and that match the component class id of 6 whole blood for the given division
		///	This implements TT 3.01 and BR 3.41
		/// </summary>
		/// <param name="bloodTypeCode"></param>
		/// <param name="rhFactorCode"></param>
		/// <param name="patientGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetCompatibleWholeBloodUnits(string bloodTypeCode, string rhFactorCode, System.Guid patientGuid, string divisionCode)
		{
			if (bloodTypeCode == null || rhFactorCode == null || patientGuid == Guid.Empty || divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException();
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompatibleWholeBloodUnits.bloodtypecode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetCompatibleWholeBloodUnits.rhfactorcode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetCompatibleWholeBloodUnits.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetCompatibleWholeBloodUnits.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = bloodTypeCode;
			prms[1].Value = rhFactorCode;
			prms[2].Value = patientGuid;
			prms[3].Value = divisionCode;

			// This will return >1 DataTable, but only one is valid
			DataSet results = Common.StoredProcedure.GetData(STOREDPROC.GetCompatibleWholeBloodUnits.StoredProcName, prms);
			DataTable dtUnits = new DataTable();
			foreach (DataTable dt in results.Tables)
			{
				if (dt.Rows.Count > 0)
				{
					dtUnits = dt;
					break;
				}
			}  
        	return dtUnits;
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1576"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1577"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Selects "Platelet" and "Other" blood units that are compatible for a given patients abo rh
		/// This implements TT 3.06
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetCompatiblePlateletBloodUnits(string divisionCode)
		{
			if (divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			System.Data.DataTable dtRetVal = new System.Data.DataTable();
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompatiblePlateletBloodUnits.divisioncode, System.Data.SqlDbType.Char)
			};
		
			prms[0].Value = divisionCode;

			dtRetVal = Common.StoredProcedure.GetData(STOREDPROC.GetCompatiblePlateletBloodUnits.StoredProcName, prms).Tables[0];
			return Sort306ABORhTruthTables(dtRetVal);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2183"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2184"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetCompatibleOtherBloodUnits
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetCompatibleOtherBloodUnits(string divisionCode)
		{
			if (divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			System.Data.DataTable dtRetVal = new System.Data.DataTable();
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompatibleOtherBloodUnits.divisioncode, System.Data.SqlDbType.Char)
			};
		
			prms[0].Value = divisionCode;

			dtRetVal = Common.StoredProcedure.GetData(STOREDPROC.GetCompatibleOtherBloodUnits.StoredProcName, prms).Tables[0];
			return Sort306ABORhTruthTables(dtRetVal);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1581"> 
		///		<ExpectedInput>BloodTypeCode, RhFactorCode, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable of Unit Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1584"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetCompatibleNonWholeBloodNonPlasmaUnits
		/// </summary>
		/// <param name="bloodTypeCode"></param>
		/// <param name="rhFactorCode"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetCompatibleNonWholeBloodNonPlasmaUnits(string bloodTypeCode, string rhFactorCode, string divisionCode)
		{
			if (bloodTypeCode == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodTypeCode").ResString);
			}
			if (rhFactorCode == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("rhFactorCode").ResString);
			}
			if (divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			System.Data.DataTable dtRetVal = new System.Data.DataTable();
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompatibleNonWholeBloodNonPlasmaUnits.BloodTypeCode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetCompatibleNonWholeBloodNonPlasmaUnits.RhFactorCode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetCompatibleNonWholeBloodNonPlasmaUnits.DivisionCode, System.Data.SqlDbType.Char)
			};
		
			prms[0].Value = bloodTypeCode;
			prms[1].Value = rhFactorCode;
			prms[2].Value = divisionCode;

			// This will return >1 DataTable, but only one is valid
			try
			{
				dtRetVal = Common.StoredProcedure.GetData(STOREDPROC.GetCompatibleNonWholeBloodNonPlasmaUnits.StoredProcName, prms).Tables[0];
				return SortABORhTruthTables(bloodTypeCode, rhFactorCode, dtRetVal);
			}
			catch (Exception exc)
			{
				Console.WriteLine("Exc: " + exc);
				return null;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		/// <summary>
		/// Sorts the ABO RH Compatibility truth tables bases on patients abo and rh values.
		/// </summary>
		/// <param name="bloodTypeCode"></param>
		/// <param name="rhFactorCode"></param>
		/// <param name="dtUnSorted"></param>
		/// <returns></returns>
		private static System.Data.DataTable SortABORhTruthTables(string bloodTypeCode, string rhFactorCode, System.Data.DataTable dtUnSorted)
		{
			System.Data.DataTable dtSorted = new System.Data.DataTable();

			dtSorted = dtUnSorted.Clone();
			//Find all the rows that match the patients blood type and RhFactor Type  they should be first in the datatable
			System.Data.DataRow [] dtRowsMatchBloodTypeRhfactorType = dtUnSorted.Select("BloodTypeCode = '" + bloodTypeCode + "' AND RhFactorCode = '" + rhFactorCode + "'");
			
			if(dtRowsMatchBloodTypeRhfactorType.Length > 0)
			{
				foreach(System.Data.DataRow drSource in dtRowsMatchBloodTypeRhfactorType)
				{
					System.Data.DataRow drSortedRow = dtSorted.NewRow();
					drSortedRow.ItemArray = drSource.ItemArray;
					dtSorted.Rows.Add(drSortedRow);
				}
			}
			//Add all the other abo types that match the rh factor of the patient
			System.Data.DataRow [] dtRowsMatchRhFactorTypeNotBloodType = dtUnSorted.Select("BloodTypeCode <> '" + bloodTypeCode + "' AND RhFactorCode = '" + rhFactorCode + "'");
			
			if(dtRowsMatchRhFactorTypeNotBloodType.Length > 0)
			{
				foreach(System.Data.DataRow drSource in dtRowsMatchRhFactorTypeNotBloodType)
				{
					System.Data.DataRow drSortedRow = dtSorted.NewRow();
					drSortedRow.ItemArray = drSource.ItemArray;
					dtSorted.Rows.Add(drSortedRow);
				}
			}

			//Add all the abo types that match and the rh factor not match the patient
			System.Data.DataRow [] dtRowsMatchBloodTypeNotRhFactorType = dtUnSorted.Select("BloodTypeCode = '" + bloodTypeCode + "' AND RhFactorCode <> '" + rhFactorCode + "'");
			if(dtRowsMatchBloodTypeNotRhFactorType.Length > 0)
			{
				foreach(System.Data.DataRow drSource in dtRowsMatchBloodTypeNotRhFactorType)
				{
					System.Data.DataRow drSortedRow = dtSorted.NewRow();
					drSortedRow.ItemArray = drSource.ItemArray;
					dtSorted.Rows.Add(drSortedRow);
				}
			}

			//Add all the abo types that do not match and the rh factor that do not match the patient
			System.Data.DataRow [] dtRowsNotMatchRhFactorTypeNotBloodType = dtUnSorted.Select("BloodTypeCode <> '" + bloodTypeCode + "' AND RhFactorCode <> '" + rhFactorCode + "'");
			if(dtRowsNotMatchRhFactorTypeNotBloodType.Length > 0)
			{
				foreach(System.Data.DataRow drSource in dtRowsNotMatchRhFactorTypeNotBloodType)
				{
					System.Data.DataRow drSortedRow = dtSorted.NewRow();
					drSortedRow.ItemArray = drSource.ItemArray;
					dtSorted.Rows.Add(drSortedRow);
				}
			}
			return dtSorted;
		}


		private static System.Data.DataTable Sort306ABORhTruthTables(System.Data.DataTable dtUnSorted)
		{
			System.Data.DataTable dtSorted = new System.Data.DataTable();

			dtSorted = dtUnSorted.Clone();
			//Find all the rows that match O Negative they get added first
			AddMatchingRows("O", "N", ref dtSorted, dtUnSorted);
			//Add A Negative next
			AddMatchingRows("A", "N", ref dtSorted, dtUnSorted);
			//Add B Negative next
			AddMatchingRows("B", "N", ref dtSorted, dtUnSorted);
			//Add AB Negative next
			AddMatchingRows("AB", "N", ref dtSorted, dtUnSorted);
			//Add Mixed negative next
			AddMatchingRows("Mx", "N", ref dtSorted, dtUnSorted);
			//Add O Positive next
			AddMatchingRows("O", "P", ref dtSorted, dtUnSorted);
			//Add A positive next
			AddMatchingRows("A", "P", ref dtSorted, dtUnSorted);
			//Add B positive next
			AddMatchingRows("B", "P", ref dtSorted, dtUnSorted);
			//Add AB positive next
			AddMatchingRows("AB", "P", ref dtSorted, dtUnSorted);
			//Add Mixed positive next
			AddMatchingRows("Mx", "P", ref dtSorted, dtUnSorted);
			//Add O Mixed next
			AddMatchingRows("O", "Y", ref dtSorted, dtUnSorted);
			//Add A Mixed next
			AddMatchingRows("A", "Y", ref dtSorted, dtUnSorted);
			//Add B Mixed next
			AddMatchingRows("B", "Y", ref dtSorted, dtUnSorted);
			//Add AB Mixed next
			AddMatchingRows("AB", "Y", ref dtSorted, dtUnSorted);
			//Add Mixed Mixed finally
			return dtSorted;
		}
		/// <summary>
		/// Adds rows to the sorted datatable that match the abo and rh passed in
		/// used by Sort306ABORhTruthTables
		/// </summary>
		/// <param name="bloodTypeCode"></param>
		/// <param name="rhFactorCode"></param>
		/// <param name="dtSorted"></param>
		/// <param name="dtUnSorted"></param>
		private static void AddMatchingRows(string bloodTypeCode, string rhFactorCode, ref System.Data.DataTable dtSorted, System.Data.DataTable dtUnSorted)
		{
			System.Data.DataRow [] dtRowsMatch = dtUnSorted.Select("BloodTypeCode = '" + bloodTypeCode + "' AND RhFactorCode = '" + rhFactorCode + "'");
			
			if(dtRowsMatch.Length > 0)
			{
				foreach(System.Data.DataRow drSource in dtRowsMatch)
				{
					System.Data.DataRow drSortedRow = dtSorted.NewRow();
					drSortedRow.ItemArray = drSource.ItemArray;
					dtSorted.Rows.Add(drSortedRow);
				}
			}
		}

		/// <summary>
		/// Sorts the ABO Compatibility truth tables bases on patients abo and rh values.
		/// </summary>
		/// <param name="bloodTypeCode">bloodTypeCode</param>
		/// <param name="dtUnSorted">dtUnSorted</param>
		/// <returns>Data table</returns>
		private static System.Data.DataTable SortABOTruthTables(string bloodTypeCode, System.Data.DataTable dtUnSorted)
		{
			System.Data.DataTable dtSorted = new System.Data.DataTable();

			dtSorted = dtUnSorted.Clone();
			//Find all the rows that match the patients blood type they should be first in the datatable
			System.Data.DataRow [] dtRowsMatchBloodType = dtUnSorted.Select("BloodTypeCode = '" + bloodTypeCode + "'");
			
			if(dtRowsMatchBloodType.Length > 0)
			{
				foreach(System.Data.DataRow drSource in dtRowsMatchBloodType)
				{
					System.Data.DataRow drSortedRow = dtSorted.NewRow();
					drSortedRow.ItemArray = drSource.ItemArray;
					dtSorted.Rows.Add(drSortedRow);
				}
			}
			if(bloodTypeCode == "O")
			{
				//Add all ABO of type A next
				System.Data.DataRow [] dtRowsMatchBloodTypeA = dtUnSorted.Select("BloodTypeCode = 'A'");
				if(dtRowsMatchBloodTypeA.Length > 0)
				{
					foreach(System.Data.DataRow drSource in dtRowsMatchBloodTypeA)
					{
						System.Data.DataRow drSortedRow = dtSorted.NewRow();
						drSortedRow.ItemArray = drSource.ItemArray;
						dtSorted.Rows.Add(drSortedRow);
					}
				}
				//Add all ABO of type B next
				System.Data.DataRow [] dtRowsMatchBloodTypeB = dtUnSorted.Select("BloodTypeCode = 'B'");
				if(dtRowsMatchBloodTypeB.Length > 0)
				{
					foreach(System.Data.DataRow drSource in dtRowsMatchBloodTypeB)
					{
						System.Data.DataRow drSortedRow = dtSorted.NewRow();
						drSortedRow.ItemArray = drSource.ItemArray;
						dtSorted.Rows.Add(drSortedRow);
					}
				}
				//Add all ABO of type AB next
				System.Data.DataRow [] dtRowsMatchBloodTypeAB = dtUnSorted.Select("BloodTypeCode = 'AB'");
				if(dtRowsMatchBloodTypeAB.Length > 0)
				{
					foreach(System.Data.DataRow drSource in dtRowsMatchBloodTypeAB)
					{
						System.Data.DataRow drSortedRow = dtSorted.NewRow();
						drSortedRow.ItemArray = drSource.ItemArray;
						dtSorted.Rows.Add(drSortedRow);
					}
				}
			}
			//For all other patient ABO types you just and all the remaining ABO's next
			else
			{
				//Add all the other abo types that do not match patient
				System.Data.DataRow [] dtRowsNotMatchBloodType = dtUnSorted.Select("BloodTypeCode <> '" + bloodTypeCode + "'");
				if(dtRowsNotMatchBloodType.Length > 0)
				{
					foreach(System.Data.DataRow drSource in dtRowsNotMatchBloodType)
					{
						System.Data.DataRow drSortedRow = dtSorted.NewRow();
						drSortedRow.ItemArray = drSource.ItemArray;
						dtSorted.Rows.Add(drSortedRow);
					}
				}
			}
			return dtSorted;
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1585"> 
		///		<ExpectedInput>BloodTypeCode, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable of Unit Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1595"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetCompatiblePlasmaOnlyUnits
		/// </summary>
		/// <param name="bloodTypeCode">bloodTypeCode</param>
		/// <param name="divisionCode">divisionCode</param>
		/// <returns>Data table</returns>
		public static DataTable GetCompatibleFFPUnits(string bloodTypeCode, string divisionCode)
		{
			if (bloodTypeCode == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodTypeCode").ResString);
			}
			if (divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			System.Data.DataTable dtUnSorted = new System.Data.DataTable();
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompatibleFFPUnits.bloodtypecode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetCompatibleFFPUnits.divisioncode, System.Data.SqlDbType.Char)
			};
		
			prms[0].Value = bloodTypeCode;
			prms[1].Value = divisionCode;

			// This will return >1 DataTable, but only one is valid
			dtUnSorted = Common.StoredProcedure.GetData(STOREDPROC.GetCompatibleFFPUnits.StoredProcName, prms).Tables[0];
			
			return SortABOTruthTables(bloodTypeCode, dtUnSorted);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2185"> 
		///		<ExpectedInput>BloodTypeCode, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable of Unit Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2188"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetCompatibleCryoUnits
		/// </summary>
		/// <param name="bloodTypeCode"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetCompatibleCryoUnits(string bloodTypeCode, string divisionCode)
		{
			if (bloodTypeCode == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodTypeCode").ResString);
			}
			if (divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			
			System.Data.DataTable dtUnSorted = new System.Data.DataTable();
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCompatibleCryoUnits.bloodtypecode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetCompatibleCryoUnits.divisioncode, System.Data.SqlDbType.Char)
			};
		
			prms[0].Value = bloodTypeCode;
			prms[1].Value = divisionCode;

			// This will return >1 DataTable, but only one is valid
			dtUnSorted = Common.StoredProcedure.GetData(STOREDPROC.GetCompatibleCryoUnits.StoredProcName, prms).Tables[0];
			
			return SortABOTruthTables(bloodTypeCode, dtUnSorted);
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1596"> 
		///		<ExpectedInput>5 DataTables, UpdateFunctionId</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1598"> 
		///		<ExpectedInput>Invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SaveUnits
		/// </summary>
		/// <param name="orderedUnits"></param>
		/// <param name="orderedUnitComments"></param>
		/// <param name="orderedComponents"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="dtBloodUnitSelectionExceptions">exception generated by blood unit selection</param>
		/// <param name="dtWorkloadEvents"></param>
		/// <param name="dtBCEMessages"></param>
		/// <returns>True or False</returns>
		public static bool SaveUnits(DataTable orderedUnits, DataTable orderedUnitComments, DataTable orderedComponents, Common.UpdateFunction lastUpdateFunctionId, DataTable dtBloodUnitSelectionExceptions, DataTable dtWorkloadEvents, DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();

			sprocArray.Add(STOREDPROC.InsertOrderedUnit.StoredProcName);
			if (orderedUnits.Columns.Contains(Common.VbecsTables.OrderedUnit.LastUpdateDate))
				orderedUnits.Columns.Remove(Common.VbecsTables.OrderedUnit.LastUpdateDate);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedUnits, lastUpdateFunctionId));
            
			sprocArray.Add(STOREDPROC.InsertOrderedUnitComment.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedUnitComments, lastUpdateFunctionId));

			sprocArray.Add(STOREDPROC.UpdateOrderedComponentStatus.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedComponents,lastUpdateFunctionId));

			if (dtBloodUnitSelectionExceptions != null)
			{
				sprocArray.Add(STOREDPROC.InsertExceptionUnitSelection.StoredProcName);
				dtArray.Add(dtBloodUnitSelectionExceptions);
			}

			if (dtWorkloadEvents != null)
			{
				sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
				dtArray.Add(dtWorkloadEvents);
			}
			
			if (dtBCEMessages != null)
			{
				sprocArray.Add(STOREDPROC.HL7InsertMessageLog.StoredProcName);
				dtArray.Add(dtBCEMessages);
			}
			
			int retVal = (new Common.StoredProcedure()).TransactionalGetValue(sprocArray, dtArray);
			return (retVal == 0);
		}
	}
}
